#include "stdafx.h"
#include "DiskController.h"

//
// Constructor
//
DiskController::DiskController()
{
	m_putCount = BUFS_PER_FILE;
	m_fileBuffer = new char[FBUFSIZE];

	m_mode = IDLE;
	fCount = 0;
	f = NULL;

	packet_size = WRITESIZ;

	// Used to stop our infinite write thread
	programAlive = true;

	bufStoreIndex = 0;
	bufWriteIndex = 0;

	// Create circular buffers
	buffers = new short*[BUF_STORED];
	for(int i = 0; i < BUF_STORED; i++) {
		buffers[i] = new short[WRITESIZ];
	}

	// Circular buffer data ready events
	for(int i = 0; i < BUF_STORED; i++) {
		myEvents[i] = CreateEventA(NULL, FALSE, FALSE, NULL);
	}

	// Create our write data thread
	writeThread = AfxBeginThread(
		WriteThreadEntry,
		this,
		THREAD_PRIORITY_HIGHEST,
		0,
		CREATE_SUSPENDED);
	writeThread->m_bAutoDelete = false;
	writeThread->ResumeThread();
}

//
// Deconstructor
//
DiskController::~DiskController()
{
	reset();
	delete [] m_fileBuffer;

	programAlive = false;

	// Set all events to break waiting
	for(int i = 0; i < BUF_STORED; i++) {
		SetEvent(myEvents[i]);
	}

	// Wait for write thread to close
	DWORD result;
	result = WaitForSingleObject(writeThread->m_hThread, 250);

	// Terminate write thread
	if(result == WAIT_TIMEOUT || result == WAIT_FAILED)
		TerminateThread(writeThread->m_hThread, 0);

	// Delete our write buffers
	for(int i = 0; i < BUF_STORED; i++) {
		delete [] buffers[i];
	}
	delete [] buffers;

	delete writeThread;
}

//
// Set the file and directory name
// Do NOT include the extension, just full path with filename
//
int DiskController::setPath(char *path)
{
	// Don't allow changing the names in the middle of operation
	if(m_mode != IDLE) return 0;

	// Full path, minus the extension, we will add the extension
	//   and our own numbering scheme later
	sprintf_s(m_path, MAX_PATH, "%s", path);

	return 1;
}

//
// Thread Entry
//
UINT DiskController::WriteThreadEntry(LPVOID param) 
{
	DiskController *dc = (DiskController*)param;
	return dc->WriteThread();
}

//
// Main write to disk thread
// Infinite loop waiting for data to write to disk
//
UINT DiskController::WriteThread() 
{
	while(programAlive) 
	{
		// Wait for data to write
		WaitForSingleObject(myEvents[bufWriteIndex], INFINITE);
		if(!programAlive) {
			return 0;
		}

		put(buffers[bufWriteIndex]);

		// Increase and wrap our bufWriteIndex
		bufWriteIndex++;
		if(bufWriteIndex >= BUF_STORED) {
			bufWriteIndex = 0;
		}
	}

	return 1;
}

//
// Does any setup needed for storing
// Reset all values, attempt to open the first file
// If successful, enter storing mode, return 1
// If un-successful at any point, return 0
//
int DiskController::beginStoring(int capture_len)
{
	m_mode = IDLE;
	if(f) fclose(f);
	fCount = 0;
	m_putCount = 0;
	packet_size = capture_len * 2;

	if(getNextWriteFile()) {
		m_mode = STORING;
		return 1;
	} else { // Unable to open specified file
		return 0;
	}

	return 1;
}

//
// Does any setup needed for retrieving
// Reset all needed values, set to idle
// If successfully opened, enter read mode, return 1
// else return 0 to indicate didn't work
//
int DiskController::beginReading()
{
	m_mode = IDLE;
	if(f) fclose(f);
	fCount = 0;
	m_getCount = 0;

	if(getNextReadFile()) {
		m_mode = READING;
		return 1;
	} else {
		return 0;
	}
}

//
// Stop command
// A nice way to put the controller into an idle state
//
void DiskController::stop()
{
	reset();
}

//
// Buffer one 299008 chunk
// Set event so thread knows to write
//
int DiskController::buffer(const short *buf) 
{
	// Make a copy of data
	memcpy(buffers[bufStoreIndex], buf, packet_size);
	
	int temp = bufStoreIndex;

	// Increase and wrap index
	bufStoreIndex++;
	if(bufStoreIndex >= BUF_STORED) {
		bufStoreIndex = 0;
	}

	if(bufStoreIndex == bufWriteIndex) {
		// If this happens, we have a problem, data is about
		//   to be dropped
	}

	// Set event to trigger write
	SetEvent(myEvents[temp]);

	return 1;
}

//
// Put 299008 shorts onto the disk
// 1) If needed open a new file
// 2) Store the packet header
// 3) Store the packet info
// 
int DiskController::put(short *buf)
{
	if(m_mode != STORING) return 0;

	if(m_putCount >= BUFS_PER_FILE) {
		getNextWriteFile();
		m_putCount = 0;
	}
	
	// Dump/flush the data
	fwrite(buf, 1, packet_size, f);
	fflush(f);

	m_putCount++;
	return 1;
}

//
// Get one WRITESIZ data chunk from the file
// Returns successful(1) if data exists
// Othersize returns 0 and puts in IDLE mode
//
int DiskController::get(short *buf) 
{
	if(m_mode != READING) return 0;

	if(m_getCount >= BUFS_PER_FILE) {
		getNextReadFile();
		m_getCount = 0;
	}

	if(fread(buf, 1, packet_size, f) == packet_size) {
		m_getCount++;
		return 1;
	} else { // Read was not successful
		reset();
		return 0;
	}

	// stahp
	return 0;
}

//
// When we have written 'n' channels to a file, this function
//   is called, these are its steps
// 1) Close the current file (if one is open)
// 2) Produce the name of the next file
// 3) Open the new file
// 4) Put in the device header and the settings header
// 5) Return successfully indicating the file is ready to write to
//
int DiskController::getNextWriteFile()
{
	// My temporary file namer
	sprintf_s(m_fileName, 259, "%s%d.bin", m_path, fCount);
	fCount++;

	if(f) fclose(f);
	if(fopen_s(&f, m_fileName, "wb") != 0)
		return 0;
	setvbuf(f, m_fileBuffer, _IOFBF, FBUFSIZE);

	return 1;
}

//
// Opens the next file for reading
//
int DiskController::getNextReadFile()
{
	sprintf_s(m_fileName, 259, "%s%d.bin", m_path, fCount);
	fCount++;

	if(f) fclose(f);
	if(fopen_s(&f, m_fileName, "rb") == 0)
		return 1;
	else 
		return 0;
}

//
// Closes and resets all values
//
void DiskController::reset()
{
	m_mode = IDLE;
	if(f) fclose(f);
	f = NULL;
	fCount = 0;
	m_putCount = m_getCount = 0;
}